--[[

NoteCase Pro script
	See http://www.notecasepro.com/

SCRIPT NAME: Auto-IconListedNotes.lua

	v. 1.0 

PURPOSE: The Auto-IconListedNotes.lua script applies to all listed notes the icons designated by the user's settings for auto-icon assignments that are located on Page 08: Icons Settings of the General Settings dialog. If Auto-Icons is not active, the script will exit with an error message. This script requires that NoteCase Pro be version 4.3.7 or more recent and will produce an error message with older versions. 

USAGE INSTRUCTIONS: List the notes to which you want to apply the icons. Then execute this script. (You can process all  notes in the document by using the List All Notes action found on the Notes Menu.  By processing only listed notes rather than all notes in the document, the script enables you to process less than all notes in the document.) 

CONFIGURATION: No configuration required.

RIGHTS: This script's author, Paul E. Merrell, hereby waives all copyright and related or neighboring rights to this script, pursuant to the Creative Commons CC0 Universal relinquishment of rights found at http://creativecommons.org/publicdomain/zero/1.0/

--]]

-- make sure app version supports necessary scriptable program commands
local nMaj, nMin, nRev = Nc_Config_GetAppVersion()
if (nMaj < 4 or (nMaj == 4 and nMin < 3) or (nMaj == 4 and nMin == 3 and nRev < 7)) then
	Nc_GUI_MessageBox("ERROR: Script requires NcPro version >= 4.3.7")
	return
end

-- gets ID for current document
local nDocID = Nc_Doc_ID_GetCur()

-- checks if document is read-only; if so aborts
if Nc_Doc_ReadOnly_Get(nDocID) == 1 then
	Nc_GUI_InfoBox("ERROR: Cannot process read-only document. See File > Documents > Current > Read-Only", 1, "ERROR")
  return
end

-- gets auto-icon settings
local nEnabled, nDefaultIconType,  strDefaultIconValue, nParentIconType, strParentIconValue, nLeafIconType, strLeafIconValue, nForcedAttachmentIcon = Nc_Config_AutoIcon_Get()

-- aborts if auto-icon is not enabled
if nEnabled == 0 then
	Nc_GUI_InfoBox("ERROR: Auto-Icon assignments are not enabled.", 1, "ERROR")
  return
end

 -- gets list's note count
local nListCount = Nc_App_List_GetCount()
if nListCount == 0 then 
	Nc_GUI_InfoBox("ERROR: No notes are listed. Terminating execution.", 1, "ERROR")
 return
end

-- to optimize processing time, gets current view layout mode, 
-- switches to Note Pane-only mode, and disables event triggers
local nDisplayMode = Nc_GUI_View_Layout_Get()
Nc_GUI_View_Layout_Set (1)
Nc_Event_EnableEvents(0)

-- processes all listed notes top to bottom
for i=1,nListCount do

	-- gets the note's ID and decrements count
	local strNoteID = Nc_App_List_Note_GetID(i-1)
	
	-- is note a leaf or parent?
	local nChildCount = Nc_Note_ChildCount(nDocID,  strNoteID)

	-- applies icon
	if nChildCount > 0 then
		Nc_Note_Icon_SetValues(nDocID, strNoteID, nParentIconType, strParentIconValue)
	else
		Nc_Note_Icon_SetValues(nDocID, strNoteID, nLeafIconType, strLeafIconValue)
	end -- if nChildCount

end -- for i=1

-- restores view mode
Nc_GUI_View_Layout_Set(nDisplayMode)

-- reenables event triggers
Nc_Event_EnableEvents(1)

